function nrrd_volume_writer(filename, volume, spacing, origin, encoding)
% nrrd_volume_writer(filename, volume, spacing, origin, encoding)
% encoding: raw, ascii or gzip
% vand@dtu.dk
% Inspired by
% https://se.mathworks.com/matlabcentral/fileexchange/48621-nrrdwriter-filename-matrix-pixelspacing-origin-encoding

% matrix = permute(matrix, [2 1 3]); % so we undo permute of index in nrrdreader

if nargin<5 || isempty(encoding)
    encoding = 'ascii';
end
datatype = class(volume); % matlabs 'single' should be changed to 'float'

fid = fopen(filename, 'wb');

% Header
fprintf(fid,'NRRD0004\n');      % NRRD type 4
fprintf(fid,['type: ', class(volume), '\n']);
fprintf(fid,'dimension: 3\n');
fprintf(fid,'space: left-posterior-superior\n');
fprintf(fid,['sizes: ', num2str(size(volume)), '\n']);
fprintf(fid,['space directions: (', num2str(spacing(1)), ',0,0) (0,', ...
    num2str(spacing(2)), ',0) (0,0,', num2str(spacing(3)), ')\n']);
fprintf(fid,'kinds: domain domain domain\n');
fprintf(fid,['encoding: ', encoding, '\n']);
fprintf(fid,'endian: little\n');
fprintf(fid,['space origin: (', num2str(origin(1)), ...
    ',',num2str(origin(2)),',', num2str(origin(3)),')\n']);
fprintf(fid,'\n');

switch encoding
    case 'raw'
        fwrite(fid, volume(:), datatype);
    
    case 'gzip'    
        
        % Store in a raw file before compressing
        tmpBase = tempname(pwd);
        tmpFile = [tmpBase '.gz'];
        fidTmpRaw = fopen(tmpBase, 'wb');
        assert(fidTmpRaw > 3, 'Could not open temporary file for GZIP compression');
        fwrite(fidTmpRaw, volume(:), datatype);
        fclose(fidTmpRaw);
        
        % Now we gzip our raw file
        gzip(tmpBase);
        
        % Finally, we put this info into our nrrd file (fidIn)
        fidTmpRaw = fopen(tmpFile, 'rb');
        tmp = fread(fidTmpRaw, inf, [datatype '=>' datatype]);
        cleaner = onCleanup(@() fclose(fidTmpRaw));
        fwrite (fid, tmp, datatype);
        
        % cleaning
        delete (tmpBase);
        delete (tmpFile);
              
    case 'ascii'
        fprintf(fid,'%u ',volume(:));
end

fclose(fid);

end

