﻿type Expr = | Num of int | Var of string
            | Neg of Expr | Add of Expr * Expr
            | Sub of Expr * Expr | Mul of Expr * Expr;;


#r @"C:\mrh\Forskning\Cambridge\BOOK\Programs\Chapters\Chapter10\WebCatProgs\TextProcessing.dll";;
open TextProcessing;;
open System.Text.RegularExpressions;;



let nameReg   = Regex @"\G\s*([a-zA-Z][a-zA-Z0-9]*)";;
let numberReg = Regex @"\G\s*([0-9]+)";;

let numReg        = Regex @"\G\s*((?:\053|-|)\s*[0-9]+)";;
let varReg        = Regex @"\G\s*([a-zA-z][a-zA-Z0-9]*)";;
let plusMinReg    = Regex @"\G\s*(\053|\055)";;
let addOpReg      = plusMinReg;;
let signReg       = plusMinReg;;
let mulOpReg      = Regex @"\G\s*(\052)";;
let leftParReg    = Regex @"\G\s*(\050)";;
let rightParReg   = Regex @"\G\s*(\051)";;
let eosReg        = Regex @"\G(\s*)$";;

type parser<'a> = string -> int -> ('a * int) list;;

let mutable maxPos = 0;;
let updateMaxPos pos = if pos > maxPos then maxPos <- pos;;

let token (reg: Regex) (conv: string -> 'a) =
    (fun str pos ->
         let ma = reg.Match(str,pos)
         match ma.Success with
         | false -> []
         | _     ->
            let pos1 = pos + ma.Length
            updateMaxPos pos1
            [( conv(captureSingle ma 1), pos1)]): parser<'a>;;

let emptyToken (reg: Regex) =
    (fun str pos ->
         let ma = reg.Match(str,pos)
         match ma.Success with
         | false -> []
         | _     -> let pos1 = pos+ma.Length
                    updateMaxPos pos1
                    [( (), pos1)] ): parser<unit>;;

let name   = token nameReg id;;
let number = token numberReg int;;

let numFct (str: string) = Num (int str);;
let varFct = Var;;
let addOpFct = function
    | "+" -> fun x y -> Add(x,y)
    |  _  -> fun x y -> Sub(x,y);;
let mulOpFct _ = fun x y -> Mul(x,y);;
let signFct = function
    | "+" ->  id
    |  _  ->  fun x -> Neg x;;

let num      = token numReg numFct;;
let var      = token varReg varFct;;
let addOp    = token addOpReg addOpFct;;
let mulOp    = token mulOpReg mulOpFct;;
let sign     = token signReg signFct;;
let leftPar  = emptyToken leftParReg;;
let rightPar = emptyToken rightParReg;;
let eos      = emptyToken eosReg;;


let captureHere a = (fun str pos -> [(a,pos)]): parser<'a>;;

type ParserClass() =
  member t.Bind(p: parser<'a>,
                f: 'a -> parser<'b>):parser<'b> =
    fun str pos ->
      List.collect (fun (a,apos) -> f a str apos) (p str pos)
  member bld.Zero() = (fun _ _ -> []): parser<'a>
  member bld.Return a = captureHere a
  member bld.ReturnFrom (p: parser<'a>) = p;;

let parser = ParserClass();;

let pairOf p1 p2 = parser {let! x1 = p1
                           let! x2 = p2
                           return (x1,x2)};;

let nameNumber = pairOf name number;;

let varInPars = parser {let! _ = leftPar
                        let! x = var
                        let! _ = rightPar
                        return x };;

let (<|>) (p1: parser<'a>) (p2: parser<'a>) =
    (fun str pos -> (p1 str pos) @ (p2 str pos)): parser<'a>;;

let numOrVar = num <|> var;;

let rec listOf p = parser {return []}
                   <|> parser {let! x  = p
                               let! xs = listOf p
                               return x::xs};;

let rec infixL op q =
  fun p ->  p <|> parser {let! a  = p
                          let! f  = op
                          let! b  = q
                          let  a1 = f a b
                          let  p1 = captureHere a1
                          return! p1 |> infixL op q };;


let psL = numOrVar |> infixL addOp numOrVar;;

let rec infixR p op q  =
  q <|>
  parser {let! a = p
          let! f = op
          let! b = infixR p op q
          return f a b };;


let psR = infixR numOrVar addOp numOrVar;;

let person = pairOf name (listOf number);;
let personData = listOf person;;

let rec expr    = term   |> infixL addOp term
    and term    = factor |> infixL mulOp factor
    and factor  =
      num <|> var
      <|> parser{let! f = sign
                 let! x = factor
                 return (f x)}
      <|> parser{let! _ = leftPar
                 let! x = expr
                 let! _ = rightPar
                 return x};;

let personDataString = parser {let! dt = personData
                               let! _  = eos
                               return dt };;

let exprString = parser {let! ex = expr
                         let! _  = eos
                         return ex };;

type ParseResult<'a> = ParseOk of 'a | ParseError of int;;

let parseString (p: parser<'a>) (s: string) =
    maxPos <- 0
    match p s 0 with
    | (a,_)::_ -> ParseOk a
    |   _      -> ParseError maxPos;;

